// ---------- app.js (PARTE 1) ----------
document.addEventListener('DOMContentLoaded', () => {

    /**
     * Muestra una notificación temporal (toast) en la parte inferior de la pantalla.
     * @param {string} message - El mensaje a mostrar.
     * @param {number} [duration=3000] - La duración en milisegundos.
     */
    function showToast(message, duration = 3000) {
        const existingToast = document.querySelector('.toast-notification');
        if (existingToast) {
            existingToast.remove();
        }

        const toast = document.createElement('div');
        toast.className = 'toast-notification';
        toast.textContent = message;
        document.body.appendChild(toast);

        setTimeout(() => toast.classList.add('show'), 100);
        setTimeout(() => {
            toast.classList.remove('show');
            setTimeout(() => toast.remove(), 500);
        }, duration);
    }

    /**
     * Configura la funcionalidad de la barra de búsqueda para filtrar las tarjetas de día.
     */
    function setupSearchFilter() {
        const searchInput = document.getElementById('date-search');
        if (!searchInput) return;

        const allCards = document.querySelectorAll('.day-card');
        const noResultsMessage = document.getElementById('no-results-message');
        const grid = document.querySelector('.day-cards-grid');
        const noRecordsContainer = document.querySelector('.no-records-container');

        searchInput.addEventListener('input', (e) => {
            const searchTerm = e.target.value.toLowerCase().trim();
            let visibleCards = 0;

            allCards.forEach(card => {
                const cardDateText = (card.querySelector('.card-date') && card.querySelector('.card-date').textContent) ? card.querySelector('.card-date').textContent.toLowerCase() : '';
                const isVisible = cardDateText.includes(searchTerm);
                card.style.display = isVisible ? 'flex' : 'none';
                if (isVisible) visibleCards++;
            });

            if (noRecordsContainer) return;

            if (noResultsMessage) noResultsMessage.style.display = visibleCards === 0 ? 'block' : 'none';
            if (grid) grid.style.display = visibleCards > 0 ? 'grid' : 'none';
        });
    }
 setupWhatsAppSharing();
    /**
     * Configura la lógica para el modal de registro de nuevas atenciones.
     * Reemplaza la versión anterior; maneja parsing seguro de montos, accesibilidad y teclado.
     */
 function setupAddRecordModal() {
    const addModal = document.getElementById('registerModal');
    const fab = document.getElementById('fab-add-record');
    if (!addModal || !fab) {
        
        return;
    }

    const serviceList = addModal.querySelector('.service-list') || document.querySelector('.service-list');
    const commissionInput = addModal.querySelector('#commissionAmount');
    const totalDisplay = addModal.querySelector('#modal-total');
    const submitBtn = addModal.querySelector('#submitRegistration');
    const closeBtn = addModal.querySelector('.close-modal-btn');

    // evita doble inicialización de listeners si la función se llama otra vez
    if (serviceList && serviceList.dataset.listenersAttached === '1') {
        console.log('setupAddRecordModal: listeners ya estaban atados, saliendo.');
        return;
    }

    const parseAmount = (raw) => {
        if (raw === null || raw === undefined) return 0;
        const s = String(raw).trim();
        if (s.length === 0) return 0;
        const cleaned = s.replace(/\s/g, '').replace(/S\/|s\/|S|s/g, '').replace(',', '.').replace(/[^0-9.\-]/g, '');
        const n = parseFloat(cleaned);
        return isNaN(n) ? 0 : n;
    };

    let isProcessingClick = false; // guard para evitar toggles dobles rápidos

    const updateTotal = () => {
        let selectedItems = [];
        if (serviceList) selectedItems = serviceList.querySelectorAll('.service-item.selected');

        const servicesTotal = Array.from(selectedItems).reduce((sum, item) => {
            let amountRaw = item.dataset.amount ?? item.getAttribute('data-amount') ?? null;
            if (!amountRaw) {
                const priceEl = item.querySelector('.service-price');
                amountRaw = priceEl ? priceEl.textContent : item.textContent;
            }
            return sum + parseAmount(amountRaw);
        }, 0);

        const commission = commissionInput ? parseAmount(commissionInput.value) : 0;
        const finalTotal = servicesTotal + commission;

        if (totalDisplay) totalDisplay.textContent = `S/ ${finalTotal.toFixed(2)}`;
        console.debug('updateTotal:', { servicesTotal, commission, finalTotal, selectedCount: selectedItems.length });
    };

    function toggleItemSelection(item) {
        if (!item) return;
        if (isProcessingClick) return; // evita toggles consecutivos
        isProcessingClick = true;
        const was = item.classList.contains('selected');
        if (was) {
            item.classList.remove('selected');
            item.setAttribute('aria-pressed', 'false');
        } else {
            item.classList.add('selected');
            item.setAttribute('aria-pressed', 'true');
        }
        updateTotal();
        console.log('toggleItemSelection ->', item.dataset.id ?? '(sin id)', 'selected=', !was);
        // libera el guard tras un pequeño delay
        setTimeout(() => { isProcessingClick = false; }, 120);
    }

    const resetModal = () => {
        if (serviceList) {
            serviceList.querySelectorAll('.service-item').forEach(it => {
                it.classList.remove('selected');
                it.setAttribute('aria-pressed', 'false');
            });
        }
        if (commissionInput) commissionInput.value = '';
        updateTotal();
    };

    const openModal = () => {
        resetModal();
        addModal.style.display = 'flex';
        addModal.offsetHeight;
        addModal.style.opacity = '1';
        const first = addModal.querySelector('.service-item');
        if (first) first.focus();
        console.log('Modal abierto');
    };
    const closeModal = () => {
        addModal.style.display = 'none';
        addModal.style.opacity = '';
        console.log('Modal cerrado');
    };

    const submitRegistration = async () => {
        const selectedServices = serviceList ? Array.from(serviceList.querySelectorAll('.service-item.selected')).map(s => s.dataset.id) : [];
        const commission = commissionInput ? parseAmount(commissionInput.value) : 0;

        if (selectedServices.length === 0 && commission <= 0) {
            showToast('Selecciona un servicio o ingresa una comisión.');
            return;
        }

        submitBtn.disabled = true;
        const spanTxt = submitBtn.querySelector('span');
        const originalText = spanTxt ? spanTxt.textContent : 'Guardando...';
        if (spanTxt) spanTxt.textContent = 'Guardando...';

        try {
            const response = await fetch('/leonel/public/atenciones/crear', {
                method: 'POST',
                headers: { 'Content-Type': 'application/json', 'Accept': 'application/json' },
                body: JSON.stringify({ servicios: selectedServices, comision: commission })
            });
            if (!response.ok) throw new Error('HTTP ' + response.status);
            const res = await response.json();
            if (res.success) {
                closeModal();
                showToast('¡Registro guardado con éxito!');
                setTimeout(() => window.location.reload(), 900);
            } else {
                showToast('Error: ' + (res.message || 'No se pudo guardar.'));
                console.error('submitRegistration error server:', res);
            }
        } catch (err) {
            console.error('submitRegistration exception:', err);
            showToast('Error de conexión.');
        } finally {
            submitBtn.disabled = false;
            if (spanTxt) spanTxt.textContent = originalText;
        }
    };

    // --- EVENTOS (solo delegación) ---
    if (serviceList) {
        serviceList.addEventListener('click', (e) => {
            const item = e.target.closest('.service-item');
            if (!item) return;
            if (addModal.style.display === 'none') return;
            toggleItemSelection(item);
        });

        serviceList.addEventListener('keydown', (e) => {
            const item = e.target.closest('.service-item');
            if (!item) return;
            if (e.key === 'Enter' || e.key === ' ') {
                e.preventDefault();
                toggleItemSelection(item);
            }
        });

        // asegúrate de que los items sean focusables/role si no lo están
        serviceList.querySelectorAll('.service-item').forEach(it => {
            if (!it.hasAttribute('tabindex')) it.setAttribute('tabindex', '0');
            if (!it.hasAttribute('role')) it.setAttribute('role', 'button');
            if (!it.hasAttribute('aria-pressed')) it.setAttribute('aria-pressed', 'false');
        });

        // marca que ya atamos listeners para evitar duplicados
        serviceList.dataset.listenersAttached = '1';
    } else {
        console.warn('setupAddRecordModal: No se encontró .service-list dentro del modal.');
    }

    // controles del modal
    fab.addEventListener('click', openModal);
    if (closeBtn) closeBtn.addEventListener('click', closeModal);
    addModal.addEventListener('click', e => (e.target === addModal) && closeModal());
    if (commissionInput) commissionInput.addEventListener('input', updateTotal);
    if (submitBtn) submitBtn.addEventListener('click', submitRegistration);

    const itemCount = serviceList ? serviceList.querySelectorAll('.service-item').length : 0;
    console.log('setupAddRecordModal: inicializado. items encontrados:', itemCount);
}

/**
 * Configura la lógica para compartir reportes DIARIOS y SEMANALES por WhatsApp.
 */
function setupWhatsAppSharing() {
    const whatsappModal = document.getElementById('whatsappModal');
    if (!whatsappModal) return;

    const modalTitle = whatsappModal.querySelector('#whatsappModalTitle');
    const closeBtn = whatsappModal.querySelector('.close-modal-btn');
    const sendBtn = whatsappModal.querySelector('#sendWhatsappBtn');
    const numberInput = whatsappModal.querySelector('#whatsappNumber');
    
    // Selectores para ambos tipos de botones
    const dailyShareButtons = document.querySelectorAll('.card-action-btn.whatsapp');
    const weeklyShareButton = document.getElementById('share-weekly-report-btn');

    let reportDate = '';
    let reportType = 'daily'; // 'daily' o 'weekly'

    const openModal = (date, type, title) => {
        reportDate = date;
        reportType = type;
        modalTitle.textContent = title; // Actualiza el título del modal
        numberInput.value = '';
        whatsappModal.style.display = 'flex';
        setTimeout(() => numberInput.focus(), 100);
    };

    const closeModal = () => {
        whatsappModal.style.display = 'none';
    };

    const sendWhatsAppMessage = async () => {
        const phoneNumber = numberInput.value.trim().replace(/\s/g, '');
        if (!/^\d{9}$/.test(phoneNumber)) {
            showToast('Por favor, ingresa un número peruano válido de 9 dígitos.');
            return;
        }

        sendBtn.disabled = true;
        const originalButtonText = sendBtn.querySelector('span').textContent;
        sendBtn.querySelector('span').textContent = 'Generando...';

        try {
            // Construye la URL y el mensaje dinámicamente según el tipo de reporte
            let url, message;
            if (reportType === 'weekly') {
                // Asumimos que tu backend espera la fecha del lunes para la ruta semanal
                url = `/leonel/public/reporte/pdf/semanal/${encodeURIComponent(reportDate)}?action=share`;
                message = `¡Hola! Te comparto el reporte semanal de BarberTrack (iniciando el ${reportDate}):\n\n`;
            } else {
                url = `/leonel/public/reporte/pdf/${encodeURIComponent(reportDate)}?action=share`;
                message = `¡Hola! Te comparto el reporte de BarberTrack del día ${reportDate}:\n\n`;
            }

            const response = await fetch(url, {
                method: 'GET',
                headers: { 'Accept': 'application/json' }
            });

            if (!response.ok) throw new Error(`Error del servidor: ${response.statusText}`);
            const json = await response.json();
            if (!json.success || !json.url) throw new Error(json.message || 'La respuesta del servidor no fue válida.');

            const pdfUrl = json.url;
            message += pdfUrl; // Añade la URL del PDF al mensaje
            
            const waUrl = `https://wa.me/51${phoneNumber}?text=${encodeURIComponent(message)}`;

            window.open(waUrl, '_blank');
            closeModal();

        } catch (err) {
            console.error('Error al compartir por WhatsApp:', err);
            showToast('Ocurrió un error al generar el PDF. Revisa la consola.');
        } finally {
            sendBtn.disabled = false;
            sendBtn.querySelector('span').textContent = originalButtonText;
        }
    };

    // Event listener para los botones de reporte DIARIO
    dailyShareButtons.forEach(button => {
        button.addEventListener('click', (e) => {
            e.preventDefault();
            const date = button.dataset.date;
            openModal(date, 'daily', 'Compartir Reporte Diario');
        });
    });

    // Event listener para el botón de reporte SEMANAL (si existe)
    if (weeklyShareButton) {
        weeklyShareButton.addEventListener('click', (e) => {
            e.preventDefault();
            
            // Calcula la fecha del lunes de la semana que acaba de terminar
            const today = new Date();
            // El domingo es 0. Restamos 6 días para obtener el lunes.
            const diff = today.getDate() - 6;
            const monday = new Date(today.setDate(diff));
            
            // Formatea la fecha a YYYY-MM-DD
            const mondayFormatted = monday.toISOString().split('T')[0];

            openModal(mondayFormatted, 'weekly', 'Compartir Reporte Semanal');
        });
    }

    closeBtn.addEventListener('click', closeModal);
    whatsappModal.addEventListener('click', (e) => (e.target === whatsappModal) && closeModal());
    sendBtn.addEventListener('click', sendWhatsAppMessage);
    numberInput.addEventListener('keypress', (e) => {
        if (e.key === 'Enter') sendWhatsAppMessage();
    });
}
// ---------- app.js (PARTE 2) ----------
    /**
     * Configura la lógica para el modal que muestra los detalles de un día.
     */
    function setupDetailsModal() {
        const detailsModal = document.getElementById('detailsModal');
        if (!detailsModal) return;

        const closeBtn = detailsModal.querySelector('.close-modal-btn');
        const modalTitle = detailsModal.querySelector('#detailsModalTitle');
        const modalBody = detailsModal.querySelector('#detailsModalBody');
        const dayCards = document.querySelectorAll('.card-content');

        let allDetails = [];
        let currentPage = 1;
        const itemsPerPage = 6;

        const renderPage = (page) => {
            currentPage = page;
            const totalPages = Math.ceil(allDetails.length / itemsPerPage);
            const pageItems = allDetails.slice((page - 1) * itemsPerPage, page * itemsPerPage);

            let content = '<ul class="service-list">';
            if (pageItems.length > 0) {
                pageItems.forEach(item => {
                    const serviceName = item.servicio_nombre || 'Servicio no especificado';
                    const serviceAmount = parseFloat(item.monto || item.monto_ganancia || 0).toFixed(2);
                    content += `<li class="service-item"><span class="service-name">${serviceName}</span><span class="service-price">S/ ${serviceAmount}</span></li>`;
                });
            } else {
                content += '<li>No se encontraron registros para esta fecha.</li>';
            }
            content += '</ul>';

            if (totalPages > 1) {
                content += `
                    <div class="pagination-controls">
                        <button class="pagination-btn" id="prevPageBtn" ${page === 1 ? 'disabled' : ''}>Anterior</button>
                        <span class="page-info">Página ${page} de ${totalPages}</span>
                        <button class="pagination-btn" id="nextPageBtn" ${page === totalPages ? 'disabled' : ''}>Siguiente</button>
                    </div>
                `;
            }
            modalBody.innerHTML = content;
        };

        const openModal = async (date) => {
            const formattedDate = new Date(date + 'T00:00:00').toLocaleDateString('es-ES', { year: 'numeric', month: 'long', day: 'numeric' });
            modalTitle.textContent = `Detalle del ${formattedDate}`;
            detailsModal.style.display = 'flex';
            modalBody.innerHTML = '<p>Cargando detalles...</p>';

            try {
                const response = await fetch(`/leonel/public/atenciones/detalle?fecha=${date}`);
                if (!response.ok) throw new Error('La respuesta del servidor no fue exitosa.');

                const result = await response.json();
                if (result.success) {
                    allDetails = Array.isArray(result.detalle) ? result.detalle : [];
                    renderPage(1);
                } else {
                    modalBody.innerHTML = `<p>${result.message || 'No se pudo cargar el detalle.'}</p>`;
                }
            } catch (error) {
                console.error('Error al cargar detalles:', error);
                modalBody.innerHTML = '<p>No se pudo cargar la información. Inténtalo de nuevo.</p>';
            }
        };

        const closeModal = () => {
            detailsModal.style.display = 'none';
            allDetails = [];
            modalBody.innerHTML = '';
        };

        if (dayCards && dayCards.length > 0) {
            dayCards.forEach(card => {
                card.addEventListener('click', (e) => {
                    const date = e.currentTarget.dataset.date;
                    if (date) openModal(date);
                });
            });
        }

        modalBody.addEventListener('click', e => {
            if (e.target && e.target.id === 'prevPageBtn') {
                if (currentPage > 1) renderPage(currentPage - 1);
            }
            if (e.target && e.target.id === 'nextPageBtn') {
                const totalPages = Math.ceil(allDetails.length / itemsPerPage);
                if (currentPage < totalPages) renderPage(currentPage + 1);
            }
        });

        closeBtn.addEventListener('click', closeModal);
        detailsModal.addEventListener('click', e => (e.target === detailsModal) && closeModal());
    }

    // ----- Inicialización -----
    // Llamamos a los setups una vez que todas las funciones están definidas.
    setupSearchFilter();
    setupAddRecordModal();
    setupDetailsModal();

}); // end DOMContentLoaded
