<?php

namespace App\Models;

use App\Core\Database;
use PDO;

class Atencion
{
    private PDO $db;

    public function __construct()
    {
        $this->db = Database::getInstance()->getConnection();
    }

    /**
     * Crea un registro completo (atención, servicios, productos) usando una transacción.
     */
    public function crearRegistroCompleto(int $userId, array $serviciosIds, float $comision): array
    {
        $this->db->beginTransaction();
        try {
            $serviciosData = [];
            if (!empty($serviciosIds)) {
                $placeholders = implode(',', array_fill(0, count($serviciosIds), '?'));
                $stmtServ = $this->db->prepare("SELECT id, monto_ganancia FROM servicios WHERE id IN ($placeholders)");
                $stmtServ->execute($serviciosIds);
                $serviciosData = $stmtServ->fetchAll(PDO::FETCH_KEY_PAIR);
            }

            $totalServicios = array_sum($serviciosData);
            $totalAtencion = $totalServicios + $comision;

            $stmtAten = $this->db->prepare(
                "INSERT INTO atenciones (usuario_id, ganancia_total_servicios, ganancia_total_comisiones, ganancia_total_atencion) 
                 VALUES (?, ?, ?, ?)"
            );
            $stmtAten->execute([$userId, $totalServicios, $comision, $totalAtencion]);
            $atencionId = $this->db->lastInsertId();

            if (!empty($serviciosData)) {
                $stmtDetalleServ = $this->db->prepare(
                    "INSERT INTO detalle_atencion_servicios (atencion_id, servicio_id, monto_ganancia_registrado) VALUES (?, ?, ?)"
                );
                foreach ($serviciosData as $id => $monto) {
                    $stmtDetalleServ->execute([$atencionId, $id, $monto]);
                }
            }

            $this->db->commit();
            return [
                'success' => true,
                'message' => 'Registro guardado exitosamente.',
                'nuevoTotalDia' => $this->getGananciaDelDia($userId)
            ];
        } catch (\Exception $e) {
            $this->db->rollBack();
            return ['success' => false, 'message' => 'Error al guardar el registro: ' . $e->getMessage()];
        }
    }

    /**
     * Calcula la ganancia total del día para un usuario.
     */
    public function getGananciaDelDia(int $userId): float
    {
        $stmt = $this->db->prepare(
            "SELECT SUM(ganancia_total_atencion) as total 
             FROM atenciones 
             WHERE usuario_id = ? AND DATE(fecha_hora) = CURDATE()"
        );
        $stmt->execute([$userId]);
        return (float) $stmt->fetchColumn();
    }

    /**
     * Obtiene un resumen de las ganancias totales agrupadas por día para un usuario.
     */
    public function getResumenGanancias(int $userId, int $limit = 15): array
    {
        $stmt = $this->db->prepare(
            "SELECT 
                DATE(fecha_hora) as fecha, 
                SUM(ganancia_total_atencion) as total_ganado,
                COUNT(id) as numero_registros
             FROM atenciones 
             WHERE usuario_id = ? 
             GROUP BY DATE(fecha_hora) 
             ORDER BY fecha DESC 
             LIMIT ?"
        );
        $stmt->bindParam(1, $userId, PDO::PARAM_INT);
        $stmt->bindParam(2, $limit, PDO::PARAM_INT);
        $stmt->execute();
        return $stmt->fetchAll();
    }

    /**
     * Obtiene el detalle de todos los servicios y comisiones para una fecha específica.
     */
    public function getDetallePorFecha(int $userId, string $fecha): array
    {
        $stmt = $this->db->prepare(
            "SELECT 
                s.nombre as servicio_nombre, 
                das.monto_ganancia_registrado as monto
             FROM detalle_atencion_servicios das
             JOIN servicios s ON das.servicio_id = s.id
             JOIN atenciones a ON das.atencion_id = a.id
             WHERE a.usuario_id = ? AND DATE(a.fecha_hora) = ?
             
             UNION ALL

             SELECT 
                'Comisión por Producto' as servicio_nombre, 
                a.ganancia_total_comisiones as monto
             FROM atenciones a
             WHERE a.usuario_id = ? AND DATE(a.fecha_hora) = ? AND a.ganancia_total_comisiones > 0"
        );
        $stmt->execute([$userId, $fecha, $userId, $fecha]);
        return $stmt->fetchAll();
    }

    /**
     * NUEVO MÉTODO: Obtiene el detalle de todos los servicios y comisiones para una semana completa.
     * @param int $userId El ID del usuario.
     * @param string $fechaLunes La fecha del lunes que inicia la semana en formato 'Y-m-d'.
     * @return array Un array con los detalles de toda la semana.
     */
    public function getDetallePorSemana(int $userId, string $fechaLunes): array
    {
        // Calcula la fecha de fin de semana (domingo)
        $fechaDomingo = (new \DateTime($fechaLunes))->modify('+6 days')->format('Y-m-d');

        $stmt = $this->db->prepare(
            "SELECT 
                s.nombre as servicio_nombre, 
                das.monto_ganancia_registrado as monto,
                DATE(a.fecha_hora) as fecha_atencion
             FROM detalle_atencion_servicios das
             JOIN servicios s ON das.servicio_id = s.id
             JOIN atenciones a ON das.atencion_id = a.id
             WHERE a.usuario_id = ? AND DATE(a.fecha_hora) BETWEEN ? AND ?
             
             UNION ALL

             SELECT 
                'Comisión por Producto' as servicio_nombre, 
                a.ganancia_total_comisiones as monto,
                DATE(a.fecha_hora) as fecha_atencion
             FROM atenciones a
             WHERE a.usuario_id = ? AND DATE(a.fecha_hora) BETWEEN ? AND ? AND a.ganancia_total_comisiones > 0
             ORDER BY fecha_atencion ASC"
        );
        $stmt->execute([$userId, $fechaLunes, $fechaDomingo, $userId, $fechaLunes, $fechaDomingo]);
        return $stmt->fetchAll();
    }

    public function getGananciaHoy(int $userId): float {
        $stmt = $this->db->prepare("SELECT SUM(ganancia_total_atencion) FROM atenciones WHERE usuario_id = ? AND DATE(fecha_hora) = CURDATE()");
        $stmt->execute([$userId]);
        return (float) $stmt->fetchColumn();
    }

    public function getGananciaSemana(int $userId): float {
        $stmt = $this->db->prepare("SELECT SUM(ganancia_total_atencion) FROM atenciones WHERE usuario_id = ? AND YEARWEEK(fecha_hora, 1) = YEARWEEK(CURDATE(), 1)");
        $stmt->execute([$userId]);
        return (float) $stmt->fetchColumn();
    }

    public function getGananciaMes(int $userId): float {
        $stmt = $this->db->prepare("SELECT SUM(ganancia_total_atencion) FROM atenciones WHERE usuario_id = ? AND MONTH(fecha_hora) = MONTH(CURDATE()) AND YEAR(fecha_hora) = YEAR(CURDATE())");
        $stmt->execute([$userId]);
        return (float) $stmt->fetchColumn();
    }
}
