<?php

namespace App\Controllers;

use App\Models\Atencion;
use TCPDF;
use Exception;

class ReportController
{
    // Configuración (mejor poner en env)
    private $waPhoneNumberId = 'REPLACE_WITH_WA_PHONE_NUMBER_ID';
    private $waAccessToken = 'REPLACE_WITH_WA_ACCESS_TOKEN';
    private $waDefaultRecipient = 'REPLACE_WITH_RECIPIENT_NUMBER_E164'; // ej. 519XXXXXXXX
    private $pdfStoragePath = __DIR__ . '/../../storage/reports'; // Asegúrate que existe y sea escribible

    // ReportController.php (solo el método generatePdf)
public function generatePdf(string $fecha)
{
    session_start();
    if (!isset($_SESSION['user_id'])) {
        http_response_code(403);
        echo json_encode(['success' => false, 'message' => 'Acceso denegado']);
        exit;
    }

    $action = $_GET['action'] ?? 'inline';

    try {
        $atencionModel = new \App\Models\Atencion();
        $detalles = $atencionModel->getDetallePorFecha($_SESSION['user_id'], $fecha);
        $userName = $_SESSION['user_name'] ?? 'Usuario';

        // --- INICIO DEL DISEÑO CORREGIDO Y UNIFICADO ---

        // 1. Se crea una clase PDF personalizada para tener control total sobre el encabezado y pie de página.
        $pdf = new class(PDF_PAGE_ORIENTATION, PDF_UNIT, PDF_PAGE_FORMAT, true, 'UTF-8', false) extends \TCPDF {
            // Encabezado personalizado con el título que solicitaste.
            public function Header() {
                $this->SetFont('helvetica', 'B', 18);
                // Celda para el título. El 0 indica que ocupa todo el ancho. 'C' lo centra.
                $this->Cell(0, 15, 'Reporte de Ganancias Leonel', 0, false, 'C', 0, '', 0, false, 'M', 'M');
                // Dibuja una línea sutil debajo del encabezado.
                $this->Line(15, 25, $this->getPageWidth() - 15, 25);
            }

            // Pie de página personalizado.
            public function Footer() {
                $this->SetY(-15);
                $this->SetFont('helvetica', 'I', 8);
                $this->Cell(0, 10, 'Página ' . $this->getAliasNumPage() . '/' . $this->getAliasNbPages(), 0, false, 'C', 0, '', 0, false, 'T', 'M');
            }
        };

        // 2. Configuración general del documento.
        $pdf->SetCreator('BarberTrack');
        $pdf->SetAuthor($userName);
        $pdf->SetTitle('Reporte de Ganancias - ' . $fecha);
        $pdf->SetMargins(15, 27, 15); // Márgenes: Izquierda, Arriba (importante para el header), Derecha.
        $pdf->SetHeaderMargin(10);
        $pdf->SetFooterMargin(10);
        $pdf->SetAutoPageBreak(true, 25); // Margen inferior para el salto de página.
        $pdf->AddPage();

        // 3. Se genera el contenido HTML sin el <h1> para evitar duplicados.
        $dateObj = new \DateTime($fecha);
        $formatter = new \IntlDateFormatter('es_ES', \IntlDateFormatter::FULL, \IntlDateFormatter::NONE);
        $fechaLarga = $formatter->format($dateObj);

        $html = '
        <style>
            p.subtitle {
                text-align: center;
                font-size: 11pt;
                color: #666;
                margin-top: 0;
            }
            table {
                width: 100%;
                border-collapse: collapse;
                font-family: "Roboto", sans-serif;
                font-size: 10pt;
            }
            th {
                background-color: #B98D58; /* Color primario de tu marca */
                color: #FFFFFF;
                font-weight: bold;
                text-align: left;
                padding: 10px;
            }
            td {
                padding: 10px;
                border-bottom: 1px solid #EAEAEA;
            }
            tr.item-row:nth-child(even) {
                background-color: #F8F5F2; /* Color de fondo suave */
            }
            td.align-right {
                text-align: right;
            }
            .total-row td {
                font-weight: bold;
                font-size: 11pt;
                border-top: 2px solid #333;
                background-color: #FDFBF7;
            }
        </style>

        <p class="subtitle">' . htmlspecialchars($fechaLarga) . '</p>
        <br><br>

        <table>
            <thead>
                <tr>
                    <th>Servicio / Concepto</th>
                    <th class="align-right">Ganancia</th>
                </tr>
            </thead>
            <tbody>';

        $total = 0;
        foreach ($detalles as $item) {
            $monto = $item['monto'] ?? ($item['monto_ganancia'] ?? 0);
            $html .= '
                <tr class="item-row">
                    <td>' . htmlspecialchars($item['servicio_nombre']) . '</td>
                    <td class="align-right">S/ ' . number_format($monto, 2) . '</td>
                </tr>';
            $total += $monto;
        }

        $html .= '
            </tbody>
            <tfoot>
                <tr class="total-row">
                    <td><b>TOTAL DEL DÍA</b></td>
                    <td class="align-right"><b>S/ ' . number_format($total, 2) . '</b></td>
                </tr>
            </tfoot>
        </table>';

        $pdf->writeHTML($html, true, false, true, false, '');

        // --- FIN DEL DISEÑO CORREGIDO ---

        // Lógica para guardar o mostrar el PDF (sin cambios)
        if ($action === 'download') {
            $pdf->Output('reporte-' . $fecha . '.pdf', 'D');
        } elseif ($action === 'inline') {
            $pdf->Output('reporte-' . $fecha . '.pdf', 'I');
        } elseif ($action === 'share') {
            $publicStorage = __DIR__ . '/../../public/storage/reports';
            if (!is_dir($publicStorage)) {
                mkdir($publicStorage, 0755, true);
            }
            $filename = 'reporte-' . $fecha . '-' . time() . '.pdf';
            $filepath = rtrim($publicStorage, '/\\') . DIRECTORY_SEPARATOR . $filename;
            $pdf->Output($filepath, 'F');

            $scheme = (!empty($_SERVER['HTTPS']) && $_SERVER['HTTPS'] !== 'off') ? 'https' : 'http';
            $host = $_SERVER['HTTP_HOST'];
            $basePath = rtrim(dirname($_SERVER['SCRIPT_NAME']), '/\\');
            $publicUrl = $scheme . '://' . $host . $basePath . '/storage/reports/' . $filename;

            header('Content-Type: application/json');
            echo json_encode(['success' => true, 'message' => 'PDF generado', 'url' => $publicUrl]);
        } else {
            http_response_code(400);
            echo json_encode(['success' => false, 'message' => 'Acción desconocida']);
        }
        exit;

    } catch (\Exception $e) {
        error_log('Error generatePdf: ' . $e->getMessage());
        http_response_code(500);
        echo json_encode(['success' => false, 'message' => 'Error al generar PDF', 'error' => $e->getMessage()]);
        exit;
    }
}
 /**
     * Genera un PDF detallado y profesional para una semana completa (Lunes a Domingo).
     * @param string $fechaLunes La fecha del lunes que inicia la semana (formato Y-m-d).
     */
    public function generateWeeklyPdf(string $fechaLunes)
    {
        session_start();
        if (!isset($_SESSION['user_id'])) {
            http_response_code(403);
            echo json_encode(['success' => false, 'message' => 'Acceso denegado']);
            exit;
        }

        $action = $_GET['action'] ?? 'inline';

        try {
            $atencionModel = new \App\Models\Atencion();
            // **IMPORTANTE**: Debes crear este método en tu modelo Atencion.php
            // Debe devolver todos los registros de la semana, incluyendo la fecha de cada uno.
            $detallesSemana = $atencionModel->getDetallePorSemana($_SESSION['user_id'], $fechaLunes);
            $userName = $_SESSION['user_name'] ?? 'Usuario';

            // --- INICIO DEL DISEÑO DEL REPORTE SEMANAL ---

            $pdf = new class(PDF_PAGE_ORIENTATION, PDF_UNIT, PDF_PAGE_FORMAT, true, 'UTF-8', false) extends \TCPDF {
                public function Header() {
                    $this->SetFont('helvetica', 'B', 18);
                    $this->Cell(0, 15, 'Reporte Semanal de Ganancias', 0, false, 'C', 0, '', 0, false, 'M', 'M');
                    $this->Line(15, 25, $this->getPageWidth() - 15, 25);
                }
                public function Footer() {
                    $this->SetY(-15);
                    $this->SetFont('helvetica', 'I', 8);
                    $this->Cell(0, 10, 'Página ' . $this->getAliasNumPage() . '/' . $this->getAliasNbPages(), 0, false, 'C', 0, '', 0, false, 'T', 'M');
                }
            };

            $pdf->SetCreator('BarberTrack');
            $pdf->SetAuthor($userName);
            $pdf->SetTitle('Reporte Semanal - ' . $fechaLunes);
            $pdf->SetMargins(15, 27, 15);
            $pdf->SetHeaderMargin(10);
            $pdf->SetFooterMargin(10);
            $pdf->SetAutoPageBreak(true, 25);
            $pdf->AddPage();

            $lunes = new \DateTime($fechaLunes);
            $domingo = (new \DateTime($fechaLunes))->modify('+6 days');
            $formatterRango = new \IntlDateFormatter('es_ES', \IntlDateFormatter::LONG, \IntlDateFormatter::NONE);
            $rangoFechas = 'Semana del ' . $formatterRango->format($lunes) . ' al ' . $formatterRango->format($domingo);

            // Agrupa los detalles por fecha para un fácil acceso
            $gananciasPorDia = [];
            foreach ($detallesSemana as $item) {
                $fecha = $item['fecha_atencion']; // Asume que tu consulta devuelve la fecha
                if (!isset($gananciasPorDia[$fecha])) {
                    $gananciasPorDia[$fecha] = [];
                }
                $gananciasPorDia[$fecha][] = $item;
            }

            $html = '
            <style>
                p.subtitle { text-align: center; font-size: 11pt; color: #666; margin-top: 0; }
                h2.day-header { font-family: "Playfair Display", serif; font-size: 14pt; color: #B98D58; border-bottom: 1px solid #EAEAEA; padding-bottom: 5px; margin-top: 20px; }
                p.no-records { font-style: italic; color: #999; padding: 10px; background-color: #F8F5F2; border-left: 3px solid #EAEAEA; }
                table { width: 100%; border-collapse: collapse; font-family: "Roboto", sans-serif; font-size: 10pt; }
                th { background-color: #333; color: #FFFFFF; font-weight: bold; text-align: left; padding: 8px; }
                td { padding: 8px; border-bottom: 1px solid #EAEAEA; }
                tr.item-row:nth-child(even) { background-color: #F8F5F2; }
                td.align-right { text-align: right; }
                .subtotal-row td { font-weight: bold; border-top: 1px solid #999; background-color: #FDFBF7; }
                .total-row td { font-weight: bold; font-size: 12pt; border-top: 2px solid #333; background-color: #FDFBF7; color: #B98D58; padding: 12px 8px; }
            </style>

            <p class="subtitle">' . htmlspecialchars($rangoFechas) . '</p>';

            $totalSemanal = 0;
            $formatterDia = new \IntlDateFormatter('es_ES', \IntlDateFormatter::FULL, \IntlDateFormatter::NONE);

            // Itera desde Lunes (0) hasta Domingo (6)
            for ($i = 0; $i < 7; $i++) {
                $fechaActual = (new \DateTime($fechaLunes))->modify("+$i days");
                $fechaActualStr = $fechaActual->format('Y-m-d');

                $html .= '<h2 class="day-header">' . $formatterDia->format($fechaActual) . '</h2>';

                if (isset($gananciasPorDia[$fechaActualStr]) && !empty($gananciasPorDia[$fechaActualStr])) {
                    $html .= '<table><thead><tr><th>Servicio / Concepto</th><th class="align-right">Ganancia</th></tr></thead><tbody>';
                    
                    $totalDiario = 0;
                    foreach ($gananciasPorDia[$fechaActualStr] as $item) {
                        $monto = $item['monto'] ?? ($item['monto_ganancia'] ?? 0);
                        $html .= '<tr class="item-row"><td>' . htmlspecialchars($item['servicio_nombre']) . '</td><td class="align-right">S/ ' . number_format($monto, 2) . '</td></tr>';
                        $totalDiario += $monto;
                    }

                    $html .= '<tr class="subtotal-row"><td align="right"><b>Subtotal del día:</b></td><td class="align-right"><b>S/ ' . number_format($totalDiario, 2) . '</b></td></tr>';
                    $html .= '</tbody></table>';
                    $totalSemanal += $totalDiario;
                } else {
                    $html .= '<p class="no-records">No se registraron ganancias este día.</p>';
                }
            }

            $html .= '<br><br><table><tr class="total-row"><td><b>TOTAL DE LA SEMANA</b></td><td class="align-right"><b>S/ ' . number_format($totalSemanal, 2) . '</b></td></tr></table>';

            $pdf->writeHTML($html, true, false, true, false, '');

            // --- FIN DEL DISEÑO ---

            if ($action === 'download') {
                $pdf->Output('reporte-semanal-' . $fechaLunes . '.pdf', 'D');
            } elseif ($action === 'inline') {
                $pdf->Output('reporte-semanal-' . $fechaLunes . '.pdf', 'I');
            } elseif ($action === 'share') {
                $publicStorage = __DIR__ . '/../../public/storage/reports';
                if (!is_dir($publicStorage)) mkdir($publicStorage, 0755, true);
                $filename = 'reporte-semanal-' . $fechaLunes . '-' . time() . '.pdf';
                $filepath = rtrim($publicStorage, '/\\') . DIRECTORY_SEPARATOR . $filename;
                $pdf->Output($filepath, 'F');

                $scheme = (!empty($_SERVER['HTTPS']) && $_SERVER['HTTPS'] !== 'off') ? 'https' : 'http';
                $host = $_SERVER['HTTP_HOST'];
                $basePath = rtrim(dirname($_SERVER['SCRIPT_NAME']), '/\\');
                $publicUrl = $scheme . '://' . $host . $basePath . '/storage/reports/' . $filename;

                header('Content-Type: application/json');
                echo json_encode(['success' => true, 'message' => 'PDF generado', 'url' => $publicUrl]);
            }
            exit;

        } catch (\Exception $e) {
            error_log('Error generateWeeklyPdf: ' . $e->getMessage());
            http_response_code(500);
            echo json_encode(['success' => false, 'message' => 'Error al generar PDF semanal', 'error' => $e->getMessage()]);
            exit;
        }
    }


    /**
     * Sube el PDF a la WhatsApp Cloud API como media luego envía el mensaje de tipo "document".
     * Retorna array con success y detalles.
     */
    private function sendDocumentToWhatsApp(string $filepath, string $recipientE164, string $filename): array
    {
        // Validaciones simples
        if (!file_exists($filepath)) {
            return ['success' => false, 'message' => 'Archivo no encontrado: ' . $filepath];
        }
        if (empty($this->waPhoneNumberId) || empty($this->waAccessToken)) {
            return ['success' => false, 'message' => 'WhatsApp API no configurada en el servidor.'];
        }

        // 1) Subir media (multipart/form-data)
        $uploadUrl = "https://graph.facebook.com/v16.0/{$this->waPhoneNumberId}/media";
        $ch = curl_init();
        $cfile = curl_file_create($filepath, 'application/pdf', $filename);
        $post = [
            'messaging_product' => 'whatsapp',
            'file' => $cfile
        ];
        curl_setopt($ch, CURLOPT_URL, $uploadUrl);
        curl_setopt($ch, CURLOPT_HTTPHEADER, ["Authorization: Bearer {$this->waAccessToken}"]);
        curl_setopt($ch, CURLOPT_POST, true);
        curl_setopt($ch, CURLOPT_POSTFIELDS, $post);
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
        $uploadResp = curl_exec($ch);
        $httpcode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
        if ($uploadResp === false) {
            $err = curl_error($ch);
            curl_close($ch);
            error_log("WhatsApp media upload cURL error: $err");
            return ['success' => false, 'message' => 'Error al subir media (curl)', 'error' => $err];
        }
        curl_close($ch);

        $uploadData = json_decode($uploadResp, true);
        if ($httpcode < 200 || $httpcode >= 300 || !isset($uploadData['id'])) {
            error_log("WhatsApp media upload failed: HTTP $httpcode - " . $uploadResp);
            return ['success' => false, 'message' => 'Fallo al subir media a WhatsApp', 'response' => $uploadData];
        }

        $mediaId = $uploadData['id'];

        // 2) Enviar mensaje con documento usando el media id
        $messageUrl = "https://graph.facebook.com/v16.0/{$this->waPhoneNumberId}/messages";
        $payload = [
            'messaging_product' => 'whatsapp',
            'to' => $recipientE164,
            'type' => 'document',
            'document' => [
                'id' => $mediaId,
                'filename' => $filename
            ]
        ];

        $ch2 = curl_init();
        curl_setopt($ch2, CURLOPT_URL, $messageUrl);
        curl_setopt($ch2, CURLOPT_HTTPHEADER, [
            "Authorization: Bearer {$this->waAccessToken}",
            "Content-Type: application/json"
        ]);
        curl_setopt($ch2, CURLOPT_POST, true);
        curl_setopt($ch2, CURLOPT_POSTFIELDS, json_encode($payload));
        curl_setopt($ch2, CURLOPT_RETURNTRANSFER, true);
        $sendResp = curl_exec($ch2);
        $httpcode2 = curl_getinfo($ch2, CURLINFO_HTTP_CODE);
        if ($sendResp === false) {
            $err2 = curl_error($ch2);
            curl_close($ch2);
            error_log("WhatsApp send message cURL error: $err2");
            return ['success' => false, 'message' => 'Error al enviar mensaje (curl)', 'error' => $err2];
        }
        curl_close($ch2);

        $sendData = json_decode($sendResp, true);
        if ($httpcode2 >= 200 && $httpcode2 < 300) {
            return ['success' => true, 'message' => 'Documento enviado a WhatsApp', 'detail' => $sendData];
        } else {
            error_log("WhatsApp send message failed: HTTP $httpcode2 - " . $sendResp);
            return ['success' => false, 'message' => 'Fallo al enviar documento por WhatsApp', 'response' => $sendData];
        }
    }

    // Mantengo tu generateExcel intacto
    public function generateExcel(string $fecha)
    {
        session_start();
        if (!isset($_SESSION['user_id'])) { exit('Acceso denegado'); }

        $atencionModel = new Atencion();
        $detalles = $atencionModel->getDetallePorFecha($_SESSION['user_id'], $fecha);

        header('Content-Type: text/csv');
        header('Content-Disposition: attachment; filename="reporte-' . $fecha . '.csv"');

        $output = fopen('php://output', 'w');
        fputcsv($output, ['Servicio', 'Ganancia']);
        $total = 0;
        foreach ($detalles as $item) {
            fputcsv($output, [$item['servicio_nombre'], $item['monto']]);
            $total += $item['monto'];
        }
        fputcsv($output, ['Total', $total]);
        fclose($output);
    }
}
